<?php
/****************************************************************************
 *  CARTAZ COM 4 PARTIDAS POR CARTAZ - VERSÃO OTIMIZADA
 ****************************************************************************/

// --------------------------------------------------
// 0) CONFIGURAÇÃO INICIAL
// --------------------------------------------------
set_time_limit(120); // Aumenta tempo limite para 2 minutos
date_default_timezone_set('America/Sao_Paulo');
ini_set('memory_limit', '512M'); // Aumenta memória disponível

// --------------------------------------------------
// 1) CONFIGURAÇÃO RÁPIDA

/****************************************************************************
 *  CARTAZ COM 4 PARTIDAS POR CARTAZ - VERSÃO MODIFICADA
 *
 *  Este script gera cartazes de jogos (1080x1350) com 4 cards por cartaz.
 *  Cada card exibe:
 *    - Logos dos times (160px nas quinas)
 *    - Nomes dos times em maiúsculo com ícone VS entre eles (truncados se necessário)
 *    - Logos dos canais (múltiplas, se houver) no rodapé do card
 *  O fundo do cartaz é carregado a partir do arquivo definido em config.json.
 *  A marca d'água (campo "logo" no JSON) é aplicada em 5 posições aleatórias, com
 *  tamanho definido igual ao das logos dos canais, e desenhada após todos os elementos,
 *  garantindo que fique por cima do fundo e dos demais itens.
 *  Além disso, a capa é processada a partir do caminho definido no JSON (chave "capa"),
 *  com a data atual, salva na pasta "cartazes" e enviada ao Telegram como a primeira imagem.
 *  Ao final, é possível baixar um ZIP com todos os cartazes.
 ****************************************************************************/

// --------------------------------------------------
// 0) CONFIGURAÇÃO DO FUSO HORÁRIO
// --------------------------------------------------
date_default_timezone_set('America/Sao_Paulo');

// --------------------------------------------------
// 1) CONFIGURAÇÃO E VARIÁVEIS GLOBAIS
// --------------------------------------------------
$configFile = __DIR__ . '/config.json';
if (!file_exists($configFile)) {
    die("Arquivo de configuração não encontrado (config.json).");
}
$config = json_decode(file_get_contents($configFile), true);
if (!is_array($config)) {
    die("JSON de configuração inválido em config.json.");
}

$font_path       = $config['font_path']      ?? __DIR__ . '/fontes/Montserrat-Bold.ttf';
$background_path = $config['background']     ?? __DIR__ . '/imagens/background_estadio.jpg';
$vs_icon_path    = $config['vs_icon']        ?? __DIR__ . '/jogos/vs.png';
// O caminho da marca d'água vem do JSON; verifique se o arquivo existe
$watermark_path  = $config['logo']           ?? '';

$width  = 1080;
$height = 1350;
$max_matches_per_poster = 4;

$header_font_size    = 50;  
$data_font_size      = 30;  
$comp_font_size      = 28;  
$teams_font_size     = 50;  
$rodape_font_size    = 22;  

$logo_teams_size = 160;
$logo_canal_size = 70;  // Usado para logos dos canais e para redimensionar a marca d'água
$logo_vs_size    = 60;

// Função para converter cor hexadecimal para RGB
function hex2rgb($hex) {
    $hex = ltrim($hex, '#');
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] .
               $hex[1] . $hex[1] .
               $hex[2] . $hex[2];
    }
    return [
        'r' => hexdec(substr($hex, 0, 2)),
        'g' => hexdec(substr($hex, 2, 2)),
        'b' => hexdec(substr($hex, 4, 2))
    ];
}

// Cores definidas no JSON
$card_bg_hex    = $config['card_background_color'] ?? "#191970";
$card_text_hex  = $config['card_text_color']         ?? "#0b0f0c";
$champ_text_hex = $config['championship_text_color']   ?? $card_text_hex;

$card_color      = hex2rgb($card_bg_hex);
$card_text_color = hex2rgb($card_text_hex);
$champ_text_color = hex2rgb($champ_text_hex);

// --------------------------------------------------
// 2) FUNÇÕES AUXILIARES
// --------------------------------------------------
function fetchImageData($url) {
    if (empty($url)) return false;
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $data = curl_exec($ch);
    curl_close($ch);
    return $data;
}

function loadImage($pathOrUrl) {
    if (stripos($pathOrUrl, "http") === 0) {
        $data = fetchImageData($pathOrUrl);
        if ($data !== false) {
            return imagecreatefromstring($data);
        }
    } else {
        if (file_exists($pathOrUrl)) {
            $ext = strtolower(pathinfo($pathOrUrl, PATHINFO_EXTENSION));
            if ($ext === "png") {
                return imagecreatefrompng($pathOrUrl);
            } else {
                return imagecreatefromjpeg($pathOrUrl);
            }
        }
    }
    return false;
}

$botToken    = $config['telegram']['botToken'] ?? '';
$chatId      = $config['telegram']['chatId']   ?? '';
$telegramAPI = "https://api.telegram.org/bot{$botToken}";

function sendTelegramPhoto($chatId, $photoPath, $caption = '') {
    global $telegramAPI;
    if (!file_exists($photoPath)) {
        error_log("Arquivo não encontrado: " . $photoPath);
        return false;
    }
    $data = [
        'chat_id' => $chatId,
        'caption' => $caption,
        'photo'   => new CURLFile($photoPath)
    ];
    $ch = curl_init("{$telegramAPI}/sendPhoto");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    $result = curl_exec($ch);
    curl_close($ch);
    return $result;
}

function truncateText($text, $font_size, $font_path, $max_width) {
    $ellipsis = '...';
    $bbox = imagettfbbox($font_size, 0, $font_path, $text);
    $width = $bbox[2] - $bbox[0];
    if ($width <= $max_width) {
        return $text;
    }
    while (strlen($text) > 0) {
        $text = substr($text, 0, -1);
        $bbox = imagettfbbox($font_size, 0, $font_path, $text . $ellipsis);
        $width = $bbox[2] - $bbox[0];
        if ($width <= $max_width) {
            return $text . $ellipsis;
        }
    }
    return $text;
}

function sendTelegramMediaGroupInChunks($chatId, $images, $caption = '') {
    global $telegramAPI;
    $chunks = array_chunk($images, 10);
    foreach ($chunks as $chunk) {
        $media = [];
        $postFields = ['chat_id' => $chatId];
        foreach ($chunk as $i => $imgPath) {
            if (!file_exists($imgPath)) continue;
            $media[] = [
                "type" => "photo",
                "media" => "attach://file$i"
            ];
            $postFields["file$i"] = new CURLFile($imgPath);
        }
        if ($caption && count($media) > 0) {
            $media[0]["caption"] = $caption;
        }
        $postFields["media"] = json_encode($media);
        $ch = curl_init("{$telegramAPI}/sendMediaGroup");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
        $result = curl_exec($ch);
        if (curl_errno($ch)) {
            error_log("Erro ao enviar media group: " . curl_error($ch));
        }
        curl_close($ch);
        sleep(2); // delay para evitar rate limit
    }
}

// --------------------------------------------------
// 3) LEITURA DO JSON DE JOGOS E ORDENAMENTO
// --------------------------------------------------
$remoteJsonUrl = 'http://185.194.205.66/jogos/jogos_' . date('Y-m-d') . '.json';
$jsonData = file_get_contents($remoteJsonUrl);
$jogos = json_decode($jsonData, true);
if (!is_array($jogos)) {
    die("JSON de jogos inválido ou vazio.");
}

$jogos_hoje   = [];
$jogos_amanha = [];

// Define as datas de hoje e amanhã (formato Y-m-d)
$hoje   = date('Y-m-d');
$amanha = date('Y-m-d', strtotime('+1 day'));

// *** Filtragem para jogos em destaque ***
// Agora são aceitos tanto "destaque=1" quanto "destaques=true"
$filterDestaque = false;
if ((isset($_GET['destaque']) && $_GET['destaque'] == '1') || (isset($_GET['destaques']) && $_GET['destaques'] === 'true')) {
    $filterDestaque = true;
}

function parseJogoDate($data_jogo) {
    $data_jogo = trim($data_jogo);
    if (preg_match('/^\d{1,2}\/\d{1,2}\/\d{4}$/', $data_jogo)) {
        $dt = DateTime::createFromFormat('d/m/Y', $data_jogo);
        if ($dt !== false) {
            return $dt->getTimestamp();
        }
    }
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $data_jogo)) {
        $dt = DateTime::createFromFormat('Y-m-d', $data_jogo);
        if ($dt !== false) {
            return $dt->getTimestamp();
        }
    }
    return strtotime($data_jogo);
}

foreach ($jogos as $jogo) {
    // Se o filtro de destaque estiver ativo, pula jogos sem destaque=true
    if ($filterDestaque && (!isset($jogo['destaque']) || $jogo['destaque'] !== true)) {
        continue;
    }
    
    if (!empty($jogo['data_jogo'])) {
        $dataStr = $jogo['data_jogo'];
        $ts = parseJogoDate($dataStr);
        if ($ts !== false) {
            $dataJogo = date('Y-m-d', $ts);
            if ($dataJogo === $hoje) {
                $jogos_hoje[] = $jogo;
            } elseif ($dataJogo === $amanha) {
                $jogos_amanha[] = $jogo;
            }
        } else {
            $dataStrUpper = strtoupper(trim($dataStr));
            if ($dataStrUpper === "HOJE") {
                $jogos_hoje[] = $jogo;
            } elseif ($dataStrUpper === "AMANHA") {
                $jogos_amanha[] = $jogo;
            } else {
                $diasPT = ["DOMINGO","SEGUNDA","TERÇA","QUARTA","QUINTA","SEXTA","SÁBADO"];
                $current_day = strtoupper($diasPT[date('w')]);
                $tomorrow_day = strtoupper($diasPT[(date('w') + 1) % 7]);
                if ($dataStrUpper === $current_day) {
                    $jogos_hoje[] = $jogo;
                } elseif ($dataStrUpper === $tomorrow_day) {
                    $jogos_amanha[] = $jogo;
                }
            }
        }
    }
}

usort($jogos_hoje, function($a, $b) {
    return strtotime($a['horario']) - strtotime($b['horario']);
});
usort($jogos_amanha, function($a, $b) {
    return strtotime($a['horario']) - strtotime($b['horario']);
});

$sendToTelegram = isset($_GET['enviar']);
$enviar = $sendToTelegram ? $_GET['enviar'] : "";

// --------------------------------------------------
// 4) FUNÇÃO PARA DESENHAR O CARTAZ COMPLETO
// --------------------------------------------------
function draw_poster($matches, $start_index, $max_matches, $dia_label) {
    global $width, $height;
    global $font_path, $background_path, $watermark_path;
    global $comp_font_size, $teams_font_size, $rodape_font_size;
    global $logo_teams_size, $logo_canal_size, $logo_vs_size;
    global $vs_icon_path;
    global $card_color, $card_text_color, $champ_text_color;
    
    $image = imagecreatetruecolor($width, $height);
    imagealphablending($image, true);
    imagesavealpha($image, true);
    
    // Desenha o background
    $bg = loadImage($background_path);
    if ($bg) {
        imagecopyresampled($image, $bg, 0, 0, 0, 0, $width, $height, imagesx($bg), imagesy($bg));
        imagedestroy($bg);
    } else {
        $bg_color = imagecolorallocate($image, 30, 30, 30);
        imagefilledrectangle($image, 0, 0, $width, $height, $bg_color);
    }
    
    // Desenha a data no topo
    if (stripos($dia_label, 'AMANHÃ') !== false) {
         $timeStamp = strtotime('+1 day');
    } else {
         $timeStamp = time();
    }
    $diasPT  = ["DOMINGO","SEGUNDA","TERÇA","QUARTA","QUINTA","SEXTA","SÁBADO"];
    $mesesPT = ["JAN","FEV","MAR","ABR","MAI","JUN","JUL","AGO","SET","OUT","NOV","DEZ"];
    $diaNum    = date('d', $timeStamp);
    $mesNum    = date('n', $timeStamp);
    $diaSemana = date('w', $timeStamp);
    $diaNome   = $diasPT[$diaSemana];
    $mesAbrev  = $mesesPT[$mesNum - 1];
    $textoDiaMes = sprintf('%02d %s', $diaNum, $mesAbrev);
    
    $fontSizeDiaMes    = 140;
    $fontSizeDiaSemana = 50;
    $white  = imagecolorallocate($image, 255, 255, 255);
    $green  = imagecolorallocate($image, 67, 195, 68);
    
    $y_text = 150;
    $bbox_diaMes = imagettfbbox($fontSizeDiaMes, 0, $font_path, $textoDiaMes);
    $textWidth = $bbox_diaMes[2] - $bbox_diaMes[0];
    $x_text = ($width - $textWidth) / 2;
    imagettftext($image, $fontSizeDiaMes, 0, $x_text, $y_text, $white, $font_path, $textoDiaMes);
    
    $bboxDia = imagettfbbox($fontSizeDiaSemana, 0, $font_path, $diaNome);
    $textWidthDia = $bboxDia[2] - $bboxDia[0];
    $padX = 40; 
    $padY = 20;
    $rect_width  = $textWidthDia + ($padX * 2);
    $rect_height = ($bboxDia[1] - $bboxDia[7]) + ($padY * 2);
    $x_rect = ($width - $rect_width) / 2;
    $y_rect = $y_text + 30;
    imagefilledrectangle($image, $x_rect, $y_rect, $x_rect + $rect_width, $y_rect + $rect_height, $green);
    imagettftext($image, $fontSizeDiaSemana, 0, $x_rect + $padX, $y_rect + $padY + ($bboxDia[1] - $bboxDia[7]), $white, $font_path, $diaNome);
    
    $current_y = $y_rect + $rect_height + 40;
    
    $card_height = 220;  
    $card_x = 50;
    $card_width = $width - 100;
    $spacing = 20;
    $gap = 20;
    
    // Desenha cada card (partida)
    for ($i = $start_index; $i < count($matches) && ($i - $start_index) < $max_matches; $i++) {
        $jogo = $matches[$i];
        $adm_card_bg_color = imagecolorallocatealpha($image, $card_color['r'], $card_color['g'], $card_color['b'], 60);
        imagefilledrectangle($image, $card_x, $current_y, $card_x + $card_width, $current_y + $card_height, $adm_card_bg_color);
        $white = imagecolorallocate($image, 255, 255, 255);
    
        $competicao = strtoupper($jogo['competicao'] ?? "CAMPEONATO");
        $horario    = $jogo['horario'] ?? "00:00";
        $line_top   = "$competicao - $horario";
        $bbox_comp = imagettfbbox($comp_font_size, 0, $font_path, $line_top);
        $tw_comp = $bbox_comp[2] - $bbox_comp[0];
        $x_comp = $card_x + ($card_width - $tw_comp)/2;
        $y_comp = $current_y + 40;
        $adm_champ = imagecolorallocate($image, $champ_text_color['r'], $champ_text_color['g'], $champ_text_color['b']);
        imagettftext($image, $comp_font_size, 0, $x_comp, $y_comp, $adm_champ, $font_path, $line_top);
    
        $time1_logo_x = $card_x + 10;
        $time1_logo_y = $current_y + $card_height - $logo_teams_size - 10;
        if (!empty($jogo['img_time1_url'])) {
            $logo1 = loadImage($jogo['img_time1_url']);
            if ($logo1) {
                imagecopyresampled($image, $logo1, $time1_logo_x, $time1_logo_y, 0, 0, 
                                   $logo_teams_size, $logo_teams_size, 
                                   imagesx($logo1), imagesy($logo1));
                imagedestroy($logo1);
            }
        }
        $time2_logo_x = $card_x + $card_width - $logo_teams_size - 10;
        $time2_logo_y = $time1_logo_y;
        if (!empty($jogo['img_time2_url'])) {
            $logo2 = loadImage($jogo['img_time2_url']);
            if ($logo2) {
                imagecopyresampled($image, $logo2, $time2_logo_x, $time2_logo_y, 0, 0, 
                                   $logo_teams_size, $logo_teams_size, 
                                   imagesx($logo2), imagesy($logo2));
                imagedestroy($logo2);
            }
        }
    
        $max_text_area_width = $card_width - (( $logo_teams_size + 5 ) * 2) - (2 * $gap);
        $vs_icon = loadImage($vs_icon_path);
        if (!$vs_icon) {
            $vs_text = "VS";
            $bbox_vs = imagettfbbox($teams_font_size, 0, $font_path, $vs_text);
            $w_vs = $bbox_vs[2] - $bbox_vs[0];
        } else {
            $w_vs = $logo_vs_size;
        }
        
        $max_team_name_width = ($max_text_area_width - $w_vs) / 2;
        $time1 = strtoupper($jogo['time1'] ?? "Time 1");
        $time2 = strtoupper($jogo['time2'] ?? "Time 2");
        $time1 = truncateText($time1, $teams_font_size, $font_path, $max_team_name_width);
        $time2 = truncateText($time2, $teams_font_size, $font_path, $max_team_name_width);
        $bbox_time1 = imagettfbbox($teams_font_size, 0, $font_path, $time1);
        $w_time1 = $bbox_time1[2] - $bbox_time1[0];
        $bbox_time2 = imagettfbbox($teams_font_size, 0, $font_path, $time2);
        $w_time2 = $bbox_time2[2] - $bbox_time2[0];
    
        $total_w = $w_time1 + $gap + $w_vs + $gap + $w_time2;
        $center_x = $card_x + ($card_width / 2);
        $start_x = $center_x - ($total_w / 2);
        $names_y = $current_y + ($card_height / 2) + 15;
        
        imagettftext($image, $teams_font_size, 0, $start_x, $names_y, $white, $font_path, $time1);
        $x_vs = $start_x + $w_time1 + $gap;
        if ($vs_icon) {
            $y_vs = $names_y - ($logo_vs_size / 2) - 25; 
            imagecopyresampled($image, $vs_icon, $x_vs, $y_vs, 0, 0, 
                               $logo_vs_size, $logo_vs_size, 
                               imagesx($vs_icon), imagesy($vs_icon));
            imagedestroy($vs_icon);
        } else {
            imagettftext($image, $teams_font_size, 0, $x_vs, $names_y, $white, $font_path, "VS");
        }
        $x_time2 = $x_vs + $w_vs + $gap;
        imagettftext($image, $teams_font_size, 0, $x_time2, $names_y, $white, $font_path, $time2);
    
        if (!empty($jogo['img_canal_url'])) {
            $canal_urls = explode(',', $jogo['img_canal_url']);
            $canal_urls = array_map('trim', $canal_urls);
            $num_canais = count($canal_urls);
            $space_between = 10;
            $total_canais_w = ($num_canais * $logo_canal_size) + ($space_between * ($num_canais - 1));
            $start_canais_x = $card_x + ($card_width - $total_canais_w) / 2;
            $canais_y = $current_y + $card_height - $logo_canal_size - 10;
            foreach ($canal_urls as $idx => $url_canal) {
                $cx = $start_canais_x + ($idx * ($logo_canal_size + $space_between));
                $canal_img = loadImage($url_canal);
                if ($canal_img) {
                    imagecopyresampled($image, $canal_img, $cx, $canais_y, 0, 0,
                                       $logo_canal_size, $logo_canal_size,
                                       imagesx($canal_img), imagesy($canal_img));
                    imagedestroy($canal_img);
                }
            }
        }
    
        $current_y += $card_height + $spacing;
    }
    
    // Desenha o rodapé
    $rodape_text = "Assine já e acompanhe os principais campeonatos do mundo!";
    $rodape_color = imagecolorallocate($image, 255, 255, 255);
    $bbox_rodape = imagettfbbox($rodape_font_size, 0, $font_path, $rodape_text);
    $tw_rodape = $bbox_rodape[2] - $bbox_rodape[0];
    $x_rodape = ($width - $tw_rodape) / 2;
    $y_rodape = $height - 30;
    imagettftext($image, $rodape_font_size, 0, $x_rodape, $y_rodape, $rodape_color, $font_path, $rodape_text);
    
    // **** INSERÇÃO DA MARCA D'ÁGUA SOBRE TODOS OS ELEMENTOS ****
    if (!empty($watermark_path)) {
        $watermark = loadImage($watermark_path);
        if ($watermark) {
            $orig_wm_width  = imagesx($watermark);
            $orig_wm_height = imagesy($watermark);
            // Redimensiona a marca d'água para ter a mesma largura que as logos dos canais
            $small_wm_width = $logo_canal_size;
            $small_wm_height = intval(($orig_wm_height / $orig_wm_width) * $small_wm_width);
            $small_watermark = imagecreatetruecolor($small_wm_width, $small_wm_height);
            imagealphablending($small_watermark, false);
            imagesavealpha($small_watermark, true);
            $transparent = imagecolorallocatealpha($small_watermark, 0, 0, 0, 127);
            imagefilledrectangle($small_watermark, 0, 0, $small_wm_width, $small_wm_height, $transparent);
            imagecopyresampled($small_watermark, $watermark, 0, 0, 0, 0, $small_wm_width, $small_wm_height, $orig_wm_width, $orig_wm_height);
            // Aplica a marca d'água 5 vezes em posições aleatórias com 80% de opacidade
            for ($i = 0; $i < 5; $i++) {
                $pos_x = rand(0, $width - $small_wm_width);
                $pos_y = rand(0, $height - $small_wm_height);
                imagecopymerge($image, $small_watermark, $pos_x, $pos_y, 0, 0, $small_wm_width, $small_wm_height, 80);
            }
            imagedestroy($small_watermark);
            imagedestroy($watermark);
        }
    }
    // ************************************************************
    
    $folder = __DIR__ . '/cartazes/';
    if (!is_dir($folder)) {
        mkdir($folder, 0777, true);
    }
    if (stripos($dia_label, 'AMANHÃ') !== false) {
         $fileDate = date('Y-m-d', strtotime('+1 day'));
    } else {
         $fileDate = date('Y-m-d');
    }
    $filename = $folder . 'poster_' . $fileDate . '_start_' . $start_index . '.png';
    imagepng($image, $filename, 9);
    imagedestroy($image);
    return $filename;
}

// --------------------------------------------------
// 5) GERA OS CARTAZES (HOJE e AMANHÃ)
// --------------------------------------------------
if ($sendToTelegram) {
    if ($enviar === 'hoje') {
        $telegramPhotosHoje = [];
        $start_index = 0;
        while ($start_index < count($jogos_hoje)) {
            $posterFile = draw_poster($jogos_hoje, $start_index, $max_matches_per_poster, "HOJE - " . date('d/m/Y'));
            $telegramPhotosHoje[] = $posterFile;
            $start_index += $max_matches_per_poster;
        }
    } elseif ($enviar === 'amanha') {
        $telegramPhotosAmanha = [];
        $start_index = 0;
        while ($start_index < count($jogos_amanha)) {
            $posterFile = draw_poster($jogos_amanha, $start_index, $max_matches_per_poster, "AMANHÃ - " . date('d/m/Y', strtotime('+1 day')));
            $telegramPhotosAmanha[] = $posterFile;
            $start_index += $max_matches_per_poster;
        }
    } else {
        // Caso não seja especificado enviar=hoje ou enviar=amanha, gera ambos
        $telegramPhotosHoje = [];
        $start_index = 0;
        while ($start_index < count($jogos_hoje)) {
            $posterFile = draw_poster($jogos_hoje, $start_index, $max_matches_per_poster, "HOJE - " . date('d/m/Y'));
            $telegramPhotosHoje[] = $posterFile;
            $start_index += $max_matches_per_poster;
        }
    
        $telegramPhotosAmanha = [];
        $start_index = 0;
        while ($start_index < count($jogos_amanha)) {
            $posterFile = draw_poster($jogos_amanha, $start_index, $max_matches_per_poster, "AMANHÃ - " . date('d/m/Y', strtotime('+1 day')));
            $telegramPhotosAmanha[] = $posterFile;
            $start_index += $max_matches_per_poster;
        }
    }
} else {
    // Para geração sem envio (ex.: download ZIP), gera ambos
    $telegramPhotosHoje = [];
    $start_index = 0;
    while ($start_index < count($jogos_hoje)) {
        $posterFile = draw_poster($jogos_hoje, $start_index, $max_matches_per_poster, "HOJE - " . date('d/m/Y'));
        $telegramPhotosHoje[] = $posterFile;
        $start_index += $max_matches_per_poster;
    }

    $telegramPhotosAmanha = [];
    $start_index = 0;
    while ($start_index < count($jogos_amanha)) {
        $posterFile = draw_poster($jogos_amanha, $start_index, $max_matches_per_poster, "AMANHÃ - " . date('d/m/Y', strtotime('+1 day')));
        $telegramPhotosAmanha[] = $posterFile;
        $start_index += $max_matches_per_poster;
    }
}

// --------------------------------------------------
// ENVIO VIA TELEGRAM (se o parâmetro "enviar" estiver definido)
// --------------------------------------------------
if ($sendToTelegram) {
    if ($enviar === 'hoje') {
         // Envia a capa primeiro e depois os cartazes de hoje
         enviarCapaComData($chatId);
         sleep(2);
         if (!empty($telegramPhotosHoje)) {
              if (count($telegramPhotosHoje) === 1) {
                  sendTelegramPhoto($chatId, $telegramPhotosHoje[0], "Cartaz de jogos - Hoje");
              } else {
                  sendTelegramMediaGroupInChunks($chatId, $telegramPhotosHoje, "Cartaz de jogos - Hoje");
              }
         }
    } elseif ($enviar === 'amanha') {
         if (!empty($telegramPhotosAmanha)) {
              if (count($telegramPhotosAmanha) === 1) {
                  sendTelegramPhoto($chatId, $telegramPhotosAmanha[0], "Cartaz de jogos - Amanhã");
              } else {
                  sendTelegramMediaGroupInChunks($chatId, $telegramPhotosAmanha, "Cartaz de jogos - Amanhã");
              }
         }
    } elseif ($enviar === 'capa') {
         enviarCapaComData($chatId);
    }
}

// --------------------------------------------------
// FUNÇÃO PARA PROCESSAR E ENVIAR A CAPA (BANNER)
// --------------------------------------------------
function enviarCapaComData($chatId) {
    global $config; // Utiliza o caminho da capa definido no JSON
    clearstatcache(); // Garante que a verificação do arquivo seja atualizada
    $capaOriginal = $config['capa'] ?? (__DIR__ . '/capa.jpg');
    if (!file_exists($capaOriginal)) {
        error_log("Arquivo capa.jpg não encontrado em: " . $capaOriginal);
        return;
    }
    $imgCapa = imagecreatefromjpeg($capaOriginal);
    if (!$imgCapa) {
        error_log("Falha ao criar imagem a partir de capa.jpg.");
        return;
    }
    $w = imagesx($imgCapa);
    $h = imagesy($imgCapa);
    $fontFile = __DIR__ . '/arial.ttf';
    if (!file_exists($fontFile)) {
        $fontFile = $GLOBALS['font_path'];
    }

    $diasPT  = ["DOMINGO", "SEGUNDA", "TERÇA", "QUARTA", "QUINTA", "SEXTA", "SÁBADO"];
    $mesesPT = ["JAN", "FEV", "MAR", "ABR", "MAI", "JUN", "JUL", "AGO", "SET", "OUT", "NOV", "DEZ"];

    $diaSemana = (int)date('w');
    $diaNum    = date('d');
    $mesNum    = (int)date('n');

    $diaNome   = $diasPT[$diaSemana];
    $mesAbrev  = $mesesPT[$mesNum - 1];
    $textoDiaMes = $diaNum . " " . $mesAbrev;

    $white = imagecolorallocate($imgCapa, 255, 255, 255);
    $green = imagecolorallocate($imgCapa, 67, 195, 68);

    $fontSizeDiaMes = 130;
    $bbox = imagettfbbox($fontSizeDiaMes, 0, $fontFile, $textoDiaMes);
    $textWidth = $bbox[2] - $bbox[0];

    $fontSizeDiaSemana = 20;
    $bboxDia = imagettfbbox($fontSizeDiaSemana, 0, $fontFile, $diaNome);
    $textWidthDia = $bboxDia[2] - $bboxDia[0];

    $padX = 40;
    $padY = 20;
    $rectWidth = $textWidthDia + ($padX * 2);
    $rectHeight = ($bboxDia[1] - $bboxDia[7]) + ($padY * 2);

    $y = 150;
    $yRect = $y + 30;

    // Parâmetro de deslocamento (positivo para direita, negativo para esquerda)
    $offsetX = isset($_GET['offsetX']) ? (int)$_GET['offsetX'] : 0;

    // Ajustando as posições com base no deslocamento
    $x = (($w - $textWidth) / 2) + $offsetX;
    $xRect = (($w - $rectWidth) / 2) + $offsetX;

    imagettftext($imgCapa, $fontSizeDiaMes, 0, $x, $y, $white, $fontFile, $textoDiaMes);

    imagefilledrectangle($imgCapa, $xRect, $yRect, $xRect + $rectWidth, $yRect + $rectHeight, $green);
    imagettftext($imgCapa, $fontSizeDiaSemana, 0, $xRect + $padX, $yRect + $padY + ($bboxDia[1] - $bboxDia[7]), $white, $fontFile, $diaNome);

    $folder = __DIR__ . '/cartazes/';
    if (!is_dir($folder)) {
        mkdir($folder, 0777, true);
    }

    $outputFile = $folder . 'capa_com_data.jpg';
    if (!imagejpeg($imgCapa, $outputFile, 90)) {
        error_log("Falha ao salvar a capa processada em: " . $outputFile);
    }
    imagedestroy($imgCapa);
    if (file_exists($outputFile)) {
        sendTelegramPhoto($chatId, $outputFile, "🗓 Confira os jogos de hoje! ⚽️ Grandes partidas, muita emoção e rivalidades em campo. 🏆 #Futebol #JogosDoDia
        
  (quem quiser alugar o bot que gera os banners automáticos, só chamar pv)");
    } else {
        error_log("Arquivo de capa processada não foi criado: " . $outputFile);
    }
}

// --------------------------------------------------
// FUNÇÃO PARA ZIPAR A PASTA "cartazes" E FAZER DOWNLOAD
// --------------------------------------------------
function zipDir($source, $destination) {
    if (!extension_loaded('zip') || !file_exists($source)) {
        return false;
    }
    $zip = new ZipArchive();
    if (!$zip->open($destination, ZipArchive::CREATE | ZipArchive::OVERWRITE)) {
        return false;
    }
    $source = realpath($source);
    if (is_dir($source)) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ($files as $file) {
            $file = realpath($file);
            if (is_dir($file)) {
                $zip->addEmptyDir(str_replace($source . DIRECTORY_SEPARATOR, '', $file . DIRECTORY_SEPARATOR));
            } else if (is_file($file)) {
                $zip->addFile($file, str_replace($source . DIRECTORY_SEPARATOR, '', $file));
            }
        }
    } else if (is_file($source)) {
        $zip->addFile($source, basename($source));
    }
    return $zip->close();
}

if (isset($_GET['download_zip'])) {
    $zipFile = __DIR__ . '/cartazes.zip';
    if (file_exists($zipFile)) {
        unlink($zipFile);
    }
    if (zipDir(__DIR__ . '/cartazes', $zipFile)) {
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="cartazes.zip"');
        header('Content-Length: ' . filesize($zipFile));
        readfile($zipFile);
        exit;
    } else {
        echo "Erro ao criar o ZIP.";
    }
}

echo "Cartazes gerados com sucesso!";
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Envio dos Cartazes</title>
</head>
<body>
    <h1>Envio dos Cartazes</h1>
    <p>Escolha uma opção:</p>
    <a href="?enviar=hoje">Enviar Jogos de Hoje</a> | 
    <a href="?enviar=amanha">Enviar Jogos de Amanhã</a> | 
    <a href="?enviar=capa">Enviar Capa</a> | 
    <a href="?download_zip=1">Baixar ZIP dos Cartazes</a>
    <br><br>
    <!-- Link para gerar apenas os jogos em destaque -->
    <p><strong>Opção Destaques:</strong></p>
    <a href="?destaques=true">Gerar Banner de Destaques</a>
</body>
</html>
